;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  ;; quick start instructions variables
  quick-start  ;; current quickstart instruction displayed in the quickstart monitor
  qs-item      ;; index of the current quickstart instruction
  qs-items     ;; list of quickstart instructions

  ;; lists used to create the various turtles
  shape-names       ;; list that holds the names of the non-sick shapes a student's turtle can have
  carnivore-shape-names ;; list of possible shapes for student carnivores
  herbivore-shape-names ;; list of possible shapes for student herbivores
  
  colors            ;; list that holds the colors used for students' turtles
  color-names       ;; list that holds the names of the colors used for students' turtles
  used-shape-colors ;; list that holds the shape-color pairs that are already being used

  ;; misc
  clock                ;; keeps track of the number of times through the go procedure (if there is at least one turtle infected)
  max-possible-codes   ;; total number of possible unique shape/color combinations
  max-possible-herbivore-codes ;;code system is specialized for different species, as herbivores and carnivores have different shapes
  max-possible-carnivore-codes
  
  plant-count          ;; how many plants are currently on the screen
  total-bites          ;; totals the number of points that all herbivores have lost from getting bit
  total-meat-starve    ;; totals the number of points that carnivores have lost from starvation
  total-veggie-starve  ;; totals the number of points that herbivores have lost from starvation
  
  initial-student-herbivores ;; when students are initially created, the first few should be herbivores, to prevent initial starvation
  start-points         ;; the number of points students start with
]

turtles-own
[ 
  carnivore?    ;; if a turtle is an herbivore or carnivore
  points        ;; total number of points a turtle has, based on eating, starving, pacts, and bites
  base-shape   ;; original shape of a turtle
  step-size    ;; the amount that a turtle will go forward in the current direction
]

patches-own 
[
  food      ;; the amount of food each patch currently contains
  max-food ;; the maximum amount of food on a patch
]

breeds
[
  androids  ;; created by the CREATE ANDROIDS button; not controlled by anyone, but can become sick and spread sickness
  students  ;; created and controlled by the clients
]

students-own
[
  user-id  ;; unique id, input by the client when they log in, to identify each student turtle
  old-color ;;when no longer starving, the student can go back to his/her original color
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

to startup
  setup
  setup-quick-start
  hubnet-set-client-interface "COMPUTER" [ "clients/Herbivore Carnivore Client.nlogo" ]
  hubnet-reset
  auto-plot-on
end

;; Initializes the display, and creates a list that contains the names of the shapes
;; used by turtles in this activity.  Also initializes the data lists.
to setup
  cp
  ct
  clear-output
  setup-vars
  setup-plot
  setup-patches
end

to rerun
  setup-plot
  setup-patches
  ask turtles [ 
    set heading (random 4) * 90
    fd random screen-size-x
    set points start-points
    set shape base-shape
  ]
end

;; initialize global variables
to setup-vars
  set clock 0

  set start-points 50
  set plant-count 0
  
  set initial-student-herbivores 0

  set shape-names ["wide wedge" "square" "car" "thin wedge" "big boat" "pickup truck"
                   "nu" "uu" "circle" "butterfly" "wolf" "sheep" "lobster" "monster"
                   "moose" "bear" "teddy bear"]

  set herbivore-shape-names ["wide wedge" "square" "thin wedge" 
                    "circle" "butterfly"
                   "moose" "teddy bear"  ]                   
                   
  set carnivore-shape-names [ "lobster" "monster" "car"
                    "bear" "big boat" "pickup truck" "nu" "uu" ]
                                      
  set colors      [white   brown   yellow   green   lime   turquoise
                   cyan   sky   blue   violet ]
  ;; adjust a few colors so they don't blend in with the red infection dot too much
  set colors lput (orange + 1) colors
  set colors lput (magenta + 0.5) colors
  set colors lput (pink + 2.5) colors
  set color-names ["white" "brown" "yellow" "green" "lime" "turquoise"
                   "cyan" "sky" "blue" "violet" "orange" "magenta" "pink"]
  set max-possible-codes (length colors * length shape-names)
  set max-possible-herbivore-codes (length colors * length herbivore-shape-names)
  set max-possible-carnivore-codes (length colors * length carnivore-shape-names)
  set used-shape-colors []
end

to setup-patches
cp
 ask random-n-of 2 patches 
    [ 
      set max-food 7000
    ]

  while [ max values-from patches [ max-food ] > 100 ]
     [ diffuse max-food 0.1 ]
  ask patches
  [ 
    set food round (max-food / 2)
    set pcolor scale-color (green + 2) food 0 100 
  ]

end

;; initialize the plot

to setup-plot
  clear-all-plots
end

to do-plot
  set-current-plot "Total Points"  ;;this plots the total points of herbivores, and of carnivores
  set-current-plot-pen ("meat-avg-points")
  ifelse (any? turtles with [ carnivore? ] )
    [ plot mean values-from turtles with [ carnivore? ] [ points ] ]
    [ plot 0 ]

  set-current-plot-pen ("veggie-avg-points")
  ifelse (any? turtles with [ not carnivore? ] )
    [ plot mean values-from turtles with [ not carnivore? ] [ points ] ]
    [ plot 0 ]
end

;; creates turtles that wander at random
to create-android-herbivores
  create-custom-androids 1
  [
    setxy (random-float screen-size-x) (random-float screen-size-y)
    set color grey
    set heading (random 4) * 90
    set carnivore? false
    set points start-points
    set base-shape "sheep"
    set shape base-shape
    set step-size 1
    set label "H"
    set label-color blue
  ]
end

;; creates turtles that wander at random
to create-android-carnivores
  create-custom-androids 1
  [
    setxy (random-float screen-size-x) (random-float screen-size-y)
    set color grey
    set heading (random 4) * 90
    set carnivore? true
    set points start-points
    set base-shape "wolf"
    set shape base-shape
    set step-size 1
    set label "C"
    set label-color blue
  ]
end

to make-plants
  let quantity 0
  ask patches
  [
    if food < max-food
    [
      set quantity 3
      set food (food + quantity)
      set pcolor scale-color (green + 2) food 0 100
      set plant-count plant-count + quantity
    ]    
  ]
end

;; give the user some information about what the setup button does so they can 
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? ("The SETUP button should only be used when starting "
              + "over with a new group (such as a new set of students) since "
              + "all data is lost.  Use the RE-RUN button for continuing with "
              + "an existing group."
              + "\n\nDo you really want to setup the model?")
  [
    user-message "Before closing this dialog, please do the following:"
                 + "\n  -Have everyone that is currently logged in, log off and "
                 + "then kick all remaining clients with the HubNet Console."
    setup
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  ;; get commands and data from the clients
  listen-clients
  every 1 [ ask students [ update-points-monitor ] ]

  ;; allow the androids to wander around the graphics window
  if wander?
    [ androids-wander ]
    
  plants-regrow 
  check-animals   
  do-plot
end

;; controls the motion of the androids
to androids-wander
  every android-delay
  [
    ask androids
    [
      ifelse (random 10 = 0)
        [ rt (90 * random 4) ]
      [ set heading uphill4 food ] 
      fd move-amount
    ]
  ]
end

to plants-regrow
  let quantity 0
  every plant-regrowth-rate
  [
    ask patches with [ (food < max-food) ]
    [
        set quantity 3
        set food (food + quantity)
        set pcolor scale-color (green + 2) food 0 100
    ]
  ]
end

;; report the amount we can move in the current direction
;; if we don't want to wrap around the screen
to-report move-amount  ;; turtle procedure
  ;; the value of either pxcor or pycor depending upon if
  ;; we are moving vertically or horizontally
  let new-pcor 0
  ;; the value of either screen-edge-x or screen-edge-y
  ;; depending upon if we are moving vertically or horizontally
  let screen-edge 0

  ifelse heading mod 180 = 0
  [
    set screen-edge screen-edge-y
    ifelse heading = 0
    [ set new-pcor pycor + step-size ]
    [ set new-pcor pycor - step-size ]
  ]
  [
    set screen-edge screen-edge-x
    ifelse heading = 90
    [ set new-pcor pxcor + step-size ]
    [ set new-pcor pxcor - step-size ]
  ]
  ;; if we would step past a patch at the edge of the graphics window,
  ;; make us step onto the edge patch
  if abs new-pcor > abs screen-edge
  [ report step-size - ( abs new-pcor - abs screen-edge ) ]
  report step-size
end

;; determines the gain and loss of points for each animal, based on eating, starving, and getting eaten.
to check-animals
  let eaten-turtle nobody
  ask turtles
  [
    set-starve-shape
    ifelse ( not carnivore? )
    [ 
      get-to-eat-veggie
     set points points * ( 100 - herbivore-starve-penalty ) / 100
    ]
    [
      set eaten-turtle ( random-one-of other-turtles-here with [ not carnivore? ] )
      if ( eaten-turtle != nobody ) and ( points-of eaten-turtle > start-points )
      [ get-to-eat-meat eaten-turtle ]
      set points points * (100 - carnivore-starve-penalty) / 100
    ]
    
   ] 
end

;; turtle procedure -- set the appropriate variables to make this turtle sick
to get-bit
  set points points + herbivore-bit-points
end

;;the carnivore's varialbes are changes, after finding an herbivore
to get-to-eat-meat [ eaten ]
  let eat-points (points-of eaten)
  if (eat-points > 0) [ 
    set points points + carnivore-eat-points
    ask eaten 
    [
      set points points + herbivore-bit-points 
    ] 
 ]
end

to get-to-eat-veggie
  let quantity 0
  
  set quantity random round (herbivore-eat-points * food  / 10) + 1
  
  set points points + quantity
  set food food - round (quantity / herbivore-eat-points)
  set pcolor scale-color (green + 2) food 0 100
end


to update-points-monitor
   hubnet-send user-id "Points" (round points)
end


;; when bitten, the herbivore shows a special shape for a very short period of time
to set-starve-shape
  ifelse ( points < start-points )
  [
    if ( shape != ("sick " + base-shape) )
    [ set shape "sick " + base-shape ]
  ]
  [
    ;; we want to check if the turtles shape is already a base-shape
    ;; to prevent flickering in the turles
    if (shape != base-shape)
      [ set shape base-shape ]
  ]
end

to make-students-herbivores      
  ask students [ 
    set carnivore? false
    set label "H"
  ]
end

to make-students-carnivores      
  ask students [ 
    set carnivore? true
    set label "C"
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to run the HubNet activity."
    "Optional: Zoom In (see Tools in the Menu Bar)"
    "Optional: Change any of the settings...."
      "If you do change the settings, press the SETUP button."
    "Press the GO button."
    "Everyone: Open up a HubNet Client on your machine and..."
      "input the IP Address of this computer, type..."
        "your user name in the user name box and press ENTER"
    "Teacher: Have the students move their turtles around to..."
      "acquaint themselves with the interface."
    "Everyone: Watch the plot of the average herbivore and carnivore points."
    
    "Teacher: To rerun the activity with the same group,..."
      "stop the model by pressing the NetLogo GO button, if it is on."
        "Change any of the settings that you would like."
    "Press the NetLogo RE-RUN button."
    "Teacher: Restart the simulation by pressing the NetLogo GO button again."
      "Infect some turtles and continue."
    
    "Teacher: To start the simulation over with a new group,..."
      "stop the model by pressing the NetLogo GO button, if it is on..."
        "and follow these instructions again from the beginning."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor

to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-prev
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message?
    [ create-new-student ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end

;; NetLogo knows what each student turtle is supposed to be
;; doing based on the tag sent by the node:
;; step-size - set the turtle's step-size
;; Up    - make the turtle move up by step-size
;; Down  - make the turtle move down by step-size
;; Right - make the turtle move right by step-size
;; Left  - make the turtle move left by step-size
;; Get a Different Turtle - change the turtle's shape and color
to execute-command [ command ]
  if command = "step-size"
  [
    ask students with [ user-id = hubnet-message-source ]
    [ set step-size hubnet-message ]
  ]
  if command = "Carnivore?" and allow-switching?
  [
    ask students with [ user-id = hubnet-message-source ]
    [ 
      set carnivore? hubnet-message
      ifelse carnivore?
      [ set label "C" ]
      [ set label "H" ]
    ]
  ]
  if command = "Up"
  [ execute-move 0 ]
  if command = "Down"
  [ execute-move 180 ]
  if command = "Right"
  [ execute-move 90 ]
  if command = "Left"
  [ execute-move 270 ]
  if command = "Change Appearance"
  [ execute-change-turtle ]
end

;; Create a turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to create-new-student
  create-custom-students 1
  [
    setup-student-vars
    send-info-to-clients
  ]
end

;; sets the turtle variables to appropriate initial values
to setup-student-vars  ;; turtle procedure
  set user-id hubnet-message-source
  set carnivore? false
  set-unique-shape-and-color
  setxy random screen-size-x random screen-size-y
  set heading random-one-of [0 90 270]
  set label "H"
  set label-color white
  set step-size 1
end

;; pick a base-shape and color for the turtle
to set-unique-shape-and-color
  ifelse carnivore?
  [
    let code random max-possible-carnivore-codes
    while [ member? code used-shape-colors and count students < max-possible-carnivore-codes ]
    [
      set code random max-possible-carnivore-codes
    ]
    set used-shape-colors (lput code used-shape-colors)
    set base-shape item (code mod length carnivore-shape-names) carnivore-shape-names
    set color item (code / length carnivore-shape-names) colors
  ]
  [
    let code random max-possible-herbivore-codes
    while [ member? code used-shape-colors and count students < max-possible-herbivore-codes ]
    [
      set code random max-possible-herbivore-codes
    ]
    set used-shape-colors (lput code used-shape-colors)
    set base-shape item (code mod length herbivore-shape-names) herbivore-shape-names
    set color item (code / length herbivore-shape-names) colors
  ]
  set shape base-shape
  set old-color color
end

;; report the string version of the turtle's color
to-report color-string [ color-value ]
  report item (position color-value colors) color-names
end

;; sends the appropriate monitor information back to the client
to send-info-to-clients
  hubnet-send user-id "You are a:" (color-string color) + " " + base-shape
  hubnet-send user-id "Located at:" ("(" + pxcor + "," + pycor + ")")
end

;; Kill the turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to remove-student
  ask students with [ user-id = hubnet-message-source ]
  [
    set used-shape-colors remove my-code used-shape-colors
    die
  ]
end

;; translates a student turtle's shape and color into a code
to-report my-code
  report (position base-shape shape-names) + (length shape-names) * (position color colors)
end

;; Cause the students to move forward step-size in new-heading's heading
to execute-move [ new-heading ]
  ask students with [ user-id = hubnet-message-source ]
  [
    set heading new-heading
    ;; don't allow students to wrap around the screen
    ;; so use move-amount instead of step-size
    jump move-amount
    hubnet-send user-id "Located at:" ("(" + pxcor + "," + pycor + ")")
  ]
end

;; this is called when a hubnet user chooses to change his/her turtle's appearance
to execute-change-turtle
  ask students with [ user-id = hubnet-message-source ]
  [
    set used-shape-colors remove my-code used-shape-colors
    show-turtle
    set-unique-shape-and-color
    hubnet-send user-id "You are a:" (color-string color) + " " + base-shape
  ]
end


; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2003 by Uri Wilensky. Updated 2004. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2003).  NetLogo HubNet Herbivore Carnivore model.
; http://ccl.northwestern.edu/netlogo/models/HubNetHerbivoreCarnivore.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2003 Uri Wilensky.  All rights reserved.
; See http://ccl.northwestern.edu/netlogo/models/HubNetHerbivoreCarnivore
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
395
99
825
550
10
10
20.0
1
10
0
0
0
0

CC-WINDOW
5
564
1043
659
Command Center

BUTTON
11
10
92
46
Setup
setup-prompt
NIL
1
T
OBSERVER
T
S

BUTTON
199
10
361
56
Go
go
T
1
T
OBSERVER
T
G

BUTTON
9
65
189
98
Create Herbivore Android
create-android-herbivores
NIL
1
T
OBSERVER
T
H

PLOT
14
262
351
459
Total Points
time
points
0.0
25.0
0.0
100.0
true
false
PENS
"meat-avg-points" 1.0 0 -8716033 true
"veggie-avg-points" 1.0 0 -16745473 true

SLIDER
200
105
347
138
android-delay
android-delay
0.0
10.0
0.4
0.1
1
NIL

SWITCH
199
66
348
99
wander?
wander?
0
1
-1000

BUTTON
104
10
189
46
Re-Run
rerun
NIL
1
T
OBSERVER
T
R

BUTTON
14
461
104
494
Clear Plots
setup-plot
NIL
1
T
OBSERVER
T
NIL

BUTTON
397
60
515
93
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

BUTTON
732
60
816
93
NEXT >>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
655
60
733
93
<<< PREV
view-prev
NIL
1
T
OBSERVER
T
NIL

MONITOR
397
10
817
59
Quick Start Instructions- More in Info Window
quick-start
0
1

BUTTON
9
107
189
140
Create Carnivore Android
create-android-carnivores
NIL
1
T
OBSERVER
T
C

SLIDER
834
101
1025
134
carnivore-eat-points
carnivore-eat-points
0
200
50
1
1
NIL

SLIDER
835
136
1026
169
herbivore-eat-points
herbivore-eat-points
1
100
10
1
1
NIL

SLIDER
834
171
1026
204
herbivore-bit-points
herbivore-bit-points
-500
0
-50
1
1
NIL

SLIDER
19
190
191
223
plant-regrowth-rate
plant-regrowth-rate
0
10
0.9
0.1
1
NIL

SLIDER
834
203
1034
236
herbivore-starve-penalty
herbivore-starve-penalty
0
10
5.0
0.1
1
%

SLIDER
834
236
1034
269
carnivore-starve-penalty
carnivore-starve-penalty
0
10
1.0
0.1
1
%

MONITOR
835
287
908
336
carnivores
count turtles with [ carnivore? ]
0
1

MONITOR
835
343
908
392
herbivores
count turtles with [ not carnivore? ]
3
1

SWITCH
21
153
190
186
allow-switching?
allow-switching?
0
1
-1000

BUTTON
199
153
375
186
NIL
make-students-herbivores
NIL
1
T
OBSERVER
T
NIL

BUTTON
199
191
375
224
NIL
make-students-carnivores
NIL
1
T
OBSERVER
T
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This model simulates the relationship between herbivores and carnivores.  This population can consist of either students, which are turtles controlled by individual students via the HubNet Client, or turtles that are generated and controlled by NetLogo, called androids, or both androids and students.

Turtles move around, searching for food.  Herbivores search for plants (green patches), while carnivores eat by finding other herbivores. Each animal loses points each turn that must be replenished by eating.  Herbivores may lose points whenever getting "bitten", that is, when coming into contact with a carnivore, which also results in increased points for the carnivore. When an animal is starving, it shows a red dot.

For further documentation, see the Participatory Simulations Guide found at http://ccl.northwestern.edu/ps/


HOW TO USE IT
-------------
QUICKSTART INSTRUCTIONS:
------------------------
Teacher: Follow these directions to run the HubNet activity.
Optional: Zoom In (see Tools in the Menu Bar)
Optional: Change any of the settings.  If you do change the settings, press the SETUP button.  Press the GO button to start the model.
Everyone: Open up a HubNet Client on your machine and input the IP Address of this computer, type your user name in the user name box and press ENTER.

Teacher: Have the students move their turtles around to acquaint themselves with the interface. Have the students all start as herbivores by first setting "allow-switching?" to OFF and hitting the "make-students-herbivores" button.
    
Teacher: To start the simulation over with a new group, stop the model by pressing the NetLogo GO button, if it is on and follow these instructions again from the beginning.

BUTTONS:
--------
SETUP - clears all turtles and patches and the plot.  This should only be pressed when starting out with a new group of users since all data is lost.
GO - runs the simulation
RE-RUN - resets all turtles.  This should be used to setup the model again for collecting more data or running the model again with the same users connected.
CREATE ANDROID HERBIVORE - adds randomly moving herbivore turtle to the simulation.
CREATE ANDROID CARNIVORE - adds randomly moving carnivore turtle to the simulation.

NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

SLIDERS:
--------
PLANT-REGROWTH-RATE - the delay time, in seconds, for plants to regrow, providing more food for herbivores
ANDROID-DELAY - the delay time, in seconds, for android movement - the higher the number, the slower the androids move
CARNIVORE-EAT-POINTS - the number of points a carnivore will get every time it can eat
HERBIVORE-EAT-POINTS - the number of points an herbivore will get every time it can eat
HERBIVORE-STARVE-PENALTY - the percentage of points an herbivore will lose every turn.
CARNIVORE-STARVE-PENALTY - the percentage of points a carnivore will lose every turn.
HERBIVORE-BIT-POINTS - the number of points an herbivore will lose if a carnivore gets some food out of it!

SWITCHES:
---------
PLANTS-REGROW? - when on, the patches will regrow plants, that is, will gain more food points as time goes on
ALLOW-SWITCHING? - when off, students remain as they are, either herbivores or carnivores. when this switch is on, the students can switch between acting as a herbivores or carnivores.

MONITORS:
---------
MEAN CARNIVORE POINTS - the number of points accumulated by all carnivores 
MEAN HERBIVORE POINTS - the number of points accumulated by all herbivores 
CARNIVORES - the number of current carnivores
HERBIVORES - the number of current herbivores

PLOTS:
------
Total Points- shows the number of collected points over time, comparing herbivores and carnivores

CLIENT INFORMATION
------------------
After logging in, the client interface will appear for the students, and if GO is pressed in NetLogo they will be assigned a turtle which will be described in the YOU ARE A: monitor.  And their current location will be shown in the LOCATED AT: monitor.  If the student doesn't like their assigned shape and/or color they can hit the CHANGE APPEARANCE button at any time to change to another random appearance.

The student controls the movement of their turtle with the UP, DOWN, LEFT, and RIGHT buttons and the STEP-SIZE slider.  Clicking any of the directional buttons will cause their turtle to move in the respective direction a distance of STEP-SIZE.

The students can watch the progress of the disease in the NUMBER SICK plot which is identical to the plot of the same name in NetLogo.


THINGS TO NOTICE
----------------
Android sheep tend to go toward areas of greater food.  In this model, carnivores know to go to those locations as well.

Students, when given the opportunity to change from herbivore to carnivore, may find it quite beneficial to make a change to carnivore.  However, too many students may realize this, make the change, and ultimately starve.  Students will have to figure out that what may seem beneficial to them will ultimately hurt themselves, as well as everyone else in the environment.

Carnivores may find it better to form packs.  Herbivores, however, may learn to stay more scattered, so as to not become easy prey to a large group of carnivores.


THINGS TO TRY
-------------
Use the model with the entire class to serve as an introduction to the topic.  Then have students use the NetLogo model individually, in a computer lab, to explore the effects of the various parameters.  Discuss what they find, observe, and can conclude from this model.

Try changing the amount of points that a carnivore or herbivore gets for eating or being eaten. You will find that it change change your results dramatically. Also, try changing the percentage of energy that animals expend each turn. 


EXTENDING THE MODEL
-------------------
Try making the androids smarter. 


CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (2004).  HubNet Herbivore Carnivore model. http://ccl.northwestern.edu/netlogo/models/HubNetHerbivoreCarnivore. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

@#$#@#$#@
default
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184

bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20

big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250

butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52

circle
false
0
Circle -7566196 true true 30 30 240

horse
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 65 67 74 46 79 65 93 49 88 73 65 67

lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81

monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34

moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66

nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249

pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136

sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92

sick android
false
0
Rectangle -7566196 true true 105 74 210 239
Polygon -7566196 true true 104 78 34 129 47 148 114 89
Polygon -7566196 true true 198 81 274 108 258 142 192 104
Polygon -7566196 true true 115 239 115 289 133 289 133 237
Polygon -7566196 true true 176 235 176 287 192 287 192 234
Rectangle -7566196 true true 119 12 194 73
Rectangle -16777216 true false 129 22 147 36
Rectangle -16777216 true false 164 23 184 37
Rectangle -16777216 true false 151 113 163 125
Rectangle -16777216 true false 153 142 164 154
Rectangle -16777216 true false 154 171 166 184
Circle -65536 true false 4 125 112

sick bear
false
0
Rectangle -7566196 true true 106 35 181 92
Circle -7566196 true true 110 17 18
Circle -7566196 true true 128 17 19
Rectangle -7566196 true true 80 58 111 89
Rectangle -7566196 true true 98 95 101 95
Rectangle -7566196 true true 92 93 192 104
Rectangle -7566196 true true 83 101 199 114
Rectangle -7566196 true true 78 107 205 130
Rectangle -7566196 true true 74 122 209 240
Rectangle -7566196 true true 77 236 204 249
Rectangle -7566196 true true 106 89 180 104
Rectangle -7566196 true true 80 247 198 258
Rectangle -7566196 true true 85 255 108 281
Rectangle -7566196 true true 169 252 195 283
Rectangle -7566196 true true 158 276 171 283
Rectangle -7566196 true true 74 276 90 281
Circle -7566196 true true 188 201 35
Circle -16777216 true false 111 40 13
Rectangle -7566196 true true 51 163 87 184
Rectangle -7566196 true true 45 166 53 169
Rectangle -7566196 true true 42 172 52 173
Rectangle -7566196 true true 38 178 53 183
Circle -7566196 true true 72 62 20
Circle -65536 true false 57 88 136

sick big boat
false
0
Polygon -6524078 true false 1 196 43 296 193 296 297 194
Rectangle -1 true false 135 14 149 194
Polygon -7566196 true true 151 14 173 18 193 30 211 48 239 88 251 118 271 170 271 184 253 176 227 170 199 172 177 180 161 190 165 160 169 122 165 78
Polygon -7566196 true true 133 36 115 50 77 86 47 122 7 152 33 156 57 164 77 178 91 188
Rectangle -7566196 true true 30 206 234 220
Rectangle -7566196 true true 52 224 234 236
Rectangle -7566196 true true 78 240 234 250
Circle -65536 true false 1 134 123

sick butterfly
false
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8
Circle -65536 true false 167 99 128

sick car
false
15
Circle -1 false true 185 55 60
Circle -1 false true 183 186 61
Polygon -1 true true 214 52 214 22 182 26 162 38 144 74 138 102 100 120 99 161 102 201 118 246 152 267 190 275 210 251 187 240 178 200 204 182 215 181 214 118 193 112 182 98 181 72 198 52
Circle -65536 true false 19 167 82

sick circle
false
0
Circle -7566196 true true 35 35 230
Circle -65536 true false 101 104 98

sick lobster
false
0
Polygon -7566196 true true 105 106 122 238 100 259 100 276 118 275 150 240 180 275 197 275 197 258 174 238 196 105 178 77 123 77 105 106
Polygon -7566196 true true 112 103 84 78 105 60 94 14 85 57 78 19 67 59 76 83 110 118
Polygon -7566196 true true 192 120 228 85 241 57 231 22 223 59 217 17 205 61 216 79 192 108
Rectangle -7566196 true true 71 125 117 133
Rectangle -7566196 true true 73 149 119 157
Rectangle -7566196 true true 78 178 123 187
Rectangle -7566196 true true 183 131 219 138
Rectangle -7566196 true true 180 152 216 160
Rectangle -7566196 true true 173 176 217 184
Rectangle -7566196 true true 127 56 136 82
Rectangle -7566196 true true 160 55 169 81
Circle -65536 true false 90 95 118

sick monster
false
0
Rectangle -7566196 true true 77 100 213 245
Rectangle -7566196 true true 189 243 213 273
Rectangle -7566196 true true 78 242 102 271
Rectangle -7566196 true true 59 119 83 239
Rectangle -7566196 true true 203 118 225 239
Rectangle -7566196 true true 225 123 264 158
Rectangle -7566196 true true 262 124 272 131
Rectangle -7566196 true true 263 133 273 140
Rectangle -7566196 true true 263 145 271 156
Rectangle -7566196 true true 19 122 61 156
Rectangle -7566196 true true 9 125 19 130
Rectangle -7566196 true true 11 135 20 140
Rectangle -7566196 true true 9 146 20 151
Rectangle -7566196 true true 89 79 198 101
Rectangle -7566196 true true 112 35 176 82
Rectangle -16777216 true false 122 41 136 53
Rectangle -16777216 true false 149 39 167 53
Circle -1 true false 82 118 117
Polygon -16777216 true false 122 62 132 77 157 77 168 62
Polygon -1 true false 140 63 135 72 127 62
Polygon -1 true false 145 61 151 70 159 62
Polygon -7566196 true true 123 17 115 34 131 34
Polygon -7566196 true true 165 15 149 33 172 34
Circle -65536 true false 77 102 140

sick moose
false
0
Polygon -7566196 true true 226 121 90 121 60 136 61 181 106 195 210 195 255 181 255 135
Rectangle -7566196 true true 61 180 75 268
Rectangle -7566196 true true 89 187 104 275
Rectangle -7566196 true true 210 186 225 269
Rectangle -7566196 true true 241 178 254 275
Polygon -7566196 true true 62 138 60 107 23 107 17 102 18 85 24 82 54 79 66 63 87 64 94 128
Circle -16777216 true false 63 72 12
Polygon -7566196 true true 226 120 278 138 284 185 265 188 260 142
Polygon -7566196 true true 65 65 70 50
Polygon -7566196 true true 68 64 60 49 17 49 7 41 6 12 22 40 38 40 35 14 49 39 66 39 64 16 78 66 68 64
Polygon -7566196 true true 76 64 97 17 95 38 111 38 122 16 121 38 136 38 152 11 151 38 144 47 97 47 86 66
Circle -65536 true false 79 87 115

sick nu
false
0
Rectangle -7566196 true true 16 31 60 285
Rectangle -7566196 true true 120 31 164 285
Rectangle -7566196 true true 182 31 226 285
Rectangle -7566196 true true 256 31 298 285
Rectangle -7566196 true true 214 241 274 285
Polygon -7566196 true true 60 31 118 209 120 281 60 105
Rectangle -7566196 true true 110 211 140 249
Circle -65536 true false 110 147 121

sick pickup truck
false
0
Polygon -7566196 true true 285 208 285 178 279 164 261 144 229 132 217 132 213 106 199 84 171 68 149 68 129 68 129 148 1 148 1 156 19 164 19 222 285 222 283 174 283 176
Circle -16777216 true false 40 185 71
Circle -16777216 true false 192 191 66
Circle -7566196 true true 195 194 59
Circle -7566196 true true 43 188 64
Polygon -16777216 true false 197 94 149 94 157 128 209 128 205 112 203 102 197 94
Polygon -7566196 true true 21 142 139 142 139 136 13 136
Circle -65536 true false 22 111 66

sick sheep
false
14
Rectangle -1 true false 90 75 270 225
Circle -1 true false 15 75 150
Rectangle -16777216 true true 81 225 134 286
Rectangle -16777216 true true 180 225 238 285
Circle -16777216 true true 1 88 92
Circle -65536 true false 112 46 144

sick square
false
0
Rectangle -7566196 true true 45 45 255 255
Circle -65536 true false 100 97 102

sick teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 128 80 151 69
Line -16777216 false 151 69 174 83
Polygon -65536 true false 151 208 111 173 132 153 151 169 171 154 189 169

sick thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245
Circle -65536 true false 165 140 87

sick uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286
Circle -65536 true false 43 208 93

sick wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250
Circle -65536 true false 202 65 90

sick wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192
Circle -65536 true false 130 106 106

square
false
0
Rectangle -7566196 true true 48 27 253 222

teddy bear
false
0
Circle -7566196 true true 110 21 81
Circle -7566196 true true 97 7 39
Circle -7566196 true true 171 5 39
Polygon -7566196 true true 133 88 95 117 95 225 119 247 188 248 215 224 215 116 170 91
Circle -7566196 true true 134 238 34
Polygon -7566196 true true 197 113 249 88 261 91 268 106 262 116 205 139 197 113
Polygon -7566196 true true 115 116 66 90 54 93 45 110 50 117 103 145 115 116
Polygon -7566196 true true 104 204 54 233 54 244 63 257 71 256 117 227
Polygon -7566196 true true 194 228 240 255 248 254 260 238 257 231 204 207 194 228
Circle -1 true false 124 41 20
Circle -1 true false 158 42 20
Line -16777216 false 127 75 150 85
Line -16777216 false 151 85 177 72
Polygon -1 true false 152 204 115 167 131 150 150 168 168 152 184 167

thin wedge
false
0
Polygon -7566196 true true 133 20 70 252 127 148 177 245

uu
false
0
Rectangle -7566196 true true 58 44 104 226
Rectangle -7566196 true true 88 178 194 224
Rectangle -7566196 true true 180 44 222 224
Rectangle -7566196 true true 122 74 168 284
Rectangle -7566196 true true 124 240 250 286
Rectangle -7566196 true true 240 76 284 286

wide wedge
false
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

wolf
false
0
Rectangle -7566196 true true 15 105 105 165
Rectangle -7566196 true true 45 90 105 105
Polygon -7566196 true true 60 90 83 44 104 90
Polygon -16777216 true false 67 90 82 59 97 89
Rectangle -1 true false 48 93 59 105
Rectangle -16777216 true false 51 96 55 101
Rectangle -16777216 true false 0 121 15 135
Rectangle -16777216 true false 15 136 60 151
Polygon -1 true false 15 136 23 149 31 136
Polygon -1 true false 30 151 37 136 43 151
Rectangle -7566196 true true 105 120 263 195
Rectangle -7566196 true true 108 195 259 201
Rectangle -7566196 true true 114 201 252 210
Rectangle -7566196 true true 120 210 243 214
Rectangle -7566196 true true 115 114 255 120
Rectangle -7566196 true true 128 108 248 114
Rectangle -7566196 true true 150 105 225 108
Rectangle -7566196 true true 132 214 155 270
Rectangle -7566196 true true 110 260 132 270
Rectangle -7566196 true true 210 214 232 270
Rectangle -7566196 true true 189 260 210 270
Line -7566196 true 263 127 281 155
Line -7566196 true 281 155 281 192

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
